const express = require('express');
const db = require('../db/connection');
const router = express.Router();

// Get all live battles with full details
router.get('/live', async (req, res) => {
    try {
        const query = `
            SELECT 
                b.id,
                b.status,
                b.type,
                b.genre,
                b.round_info,
                b.viewers,
                b.time_left,
                b.prize_pool,
                b.voting_end_time,
                bt.name as battle_type,
                bt.description as battle_type_description,
                
                c.id as challenge_id,
                c.challenger_id,
                c.challenged_id,
                
                -- Challenger details
                u1.username as challenger_name,
                u1.experience_points as challenger_xp,
                u1.rank as challenger_rank,
                t1.track_file as challenger_track,
                t1.votes_count as challenger_votes,
                
                -- Challenged details
                u2.username as challenged_name,
                u2.experience_points as challenged_xp,
                u2.rank as challenged_rank,
                t2.track_file as challenged_track,
                t2.votes_count as challenged_votes,
                
                -- Battle Requirements
                br.bpm_min,
                br.bpm_max,
                br.key_signature,
                br.required_samples,
                br.additional_rules,
                
                -- Tournament info if part of tournament
                tr.name as tournament_name,
                tr.prize_pool as tournament_prize
                
            FROM battles b
            JOIN challenges c ON b.challenge_id = c.id
            JOIN battle_types bt ON b.battle_type_id = bt.id
            JOIN users u1 ON c.challenger_id = u1.id
            JOIN users u2 ON c.challenged_id = u2.id
            JOIN tracks t1 ON t1.battle_id = b.id AND t1.participant_id = c.challenger_id
            JOIN tracks t2 ON t2.battle_id = b.id AND t2.participant_id = c.challenged_id
            LEFT JOIN battle_requirements br ON b.id = br.battle_id
            LEFT JOIN tournaments tr ON b.tournament_id = tr.id
            WHERE b.status IN ('LIVE', 'VOTING')
            ORDER BY b.created_at DESC
        `;
        
        const [battles] = await db.query(query);
        
        // Get judges for battles that require them
        for (let battle of battles) {
            if (battle.requires_judges) {
                const [judges] = await db.query(`
                    SELECT j.*, u.username, u.rank 
                    FROM judges j
                    JOIN users u ON j.user_id = u.id
                    WHERE j.battle_id = ?
                `, [battle.id]);
                battle.judges = judges;
            }

            // For community battles, get all participants
            if (battle.type === 'Community Battle') {
                const [participants] = await db.query(`
                    SELECT 
                        u.id,
                        u.username,
                        u.rank,
                        u.experience_points as xp,
                        t.track_file,
                        t.votes_count,
                        COUNT(DISTINCT v.id) as total_votes
                    FROM tracks t
                    JOIN users u ON t.participant_id = u.id
                    LEFT JOIN votes v ON v.track_id = t.id
                    WHERE t.battle_id = ?
                    GROUP BY t.id
                    ORDER BY total_votes DESC
                `, [battle.id]);
                battle.participants = participants;
            }

            // For tournament battles, get current matches
            if (battle.type === 'Tournament') {
                const [matches] = await db.query(`
                    SELECT 
                        m.*,
                        u1.username as producer1_name,
                        t1.track_file as producer1_track,
                        t1.votes_count as producer1_votes,
                        u2.username as producer2_name,
                        t2.track_file as producer2_track,
                        t2.votes_count as producer2_votes
                    FROM tournament_matches m
                    JOIN users u1 ON m.producer1_id = u1.id
                    JOIN users u2 ON m.producer2_id = u2.id
                    LEFT JOIN tracks t1 ON t1.battle_id = ? AND t1.participant_id = m.producer1_id
                    LEFT JOIN tracks t2 ON t2.battle_id = ? AND t2.participant_id = m.producer2_id
                    WHERE m.tournament_id = ? AND m.round = ?
                `, [battle.id, battle.id, battle.tournament_id, battle.current_round]);
                battle.currentMatches = matches;
            }
        }

        // Format response
        const formattedBattles = battles.map(battle => ({
            id: battle.id,
            type: battle.battle_type,
            typeDescription: battle.battle_type_description,
            status: battle.status,
            producer1: battle.type === 'Direct Challenge' ? {
                id: battle.challenger_id,
                name: battle.challenger_name,
                rank: battle.challenger_rank,
                xp: battle.challenger_xp,
                track: battle.challenger_track,
                votes: battle.challenger_votes
            } : null,
            producer2: battle.type === 'Direct Challenge' ? {
                id: battle.challenged_id,
                name: battle.challenged_name,
                rank: battle.challenged_rank,
                xp: battle.challenged_xp,
                track: battle.challenged_track,
                votes: battle.challenged_votes
            } : null,
            participants: battle.participants || [],
            currentMatches: battle.currentMatches || [],
            genre: battle.genre,
            roundInfo: battle.round_info,
            viewers: battle.viewers,
            timeLeft: battle.time_left,
            prizePool: battle.prize_pool,
            votingEnds: battle.voting_end_time,
            requirements: {
                bpmMin: battle.bpm_min,
                bpmMax: battle.bpm_max,
                keySignature: battle.key_signature,
                requiredSamples: battle.required_samples,
                additionalRules: battle.additional_rules
            },
            tournament: battle.tournament_name ? {
                name: battle.tournament_name,
                prizePool: battle.tournament_prize
            } : null,
            judges: battle.judges || []
        }));

        res.json(formattedBattles);
    } catch (err) {
        console.error('Error fetching battles:', err);
        res.status(500).json({ error: 'Internal server error' });
    }
});

// Get single battle details
router.get('/:id', async (req, res) => {
    try {
        const [battle] = await db.query(`
            SELECT b.*, bt.name as battle_type_name
            FROM battles b
            JOIN battle_types bt ON b.battle_type_id = bt.id
            WHERE b.id = ?
        `, [req.params.id]);

        if (!battle.length) {
            return res.status(404).json({ error: 'Battle not found' });
        }

        // Add battle details based on type...
        // Similar to the live battles query but for a single battle

        res.json(battle[0]);
    } catch (err) {
        console.error('Error fetching battle:', err);
        res.status(500).json({ error: 'Internal server error' });
    }
});

// Create a new battle
router.post('/', async (req, res) => {
    const { 
        challenge_id,
        battle_type_id,
        genre,
        round_info,
        challenger_track,
        challenged_track,
        requirements,
        tournament_id,
        judges,
        prize_pool
    } = req.body;

    try {
        await db.query('START TRANSACTION');

        // Create the battle
        const [battleResult] = await db.query(
            `INSERT INTO battles (
                challenge_id, 
                battle_type_id,
                tournament_id,
                status, 
                genre, 
                round_info, 
                viewers, 
                time_left,
                prize_pool,
                requires_judges
            ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)`,
            [
                challenge_id,
                battle_type_id,
                tournament_id,
                'LIVE',
                genre,
                round_info,
                0,
                '05:00',
                prize_pool || 0,
                judges ? true : false
            ]
        );
        
        const battleId = battleResult.insertId;

        // Get challenger and challenged IDs
        const [challengeRows] = await db.query(
            'SELECT challenger_id, challenged_id FROM challenges WHERE id = ?',
            [challenge_id]
        );
        const { challenger_id, challenged_id } = challengeRows[0];

        // Create tracks
        await db.query(
            'INSERT INTO tracks (battle_id, participant_id, track_file) VALUES (?, ?, ?), (?, ?, ?)',
            [
                battleId, challenger_id, challenger_track,
                battleId, challenged_id, challenged_track
            ]
        );

        // Add battle requirements if provided
        if (requirements) {
            await db.query(
                `INSERT INTO battle_requirements (
                    battle_id, bpm_min, bpm_max, key_signature, required_samples, additional_rules
                ) VALUES (?, ?, ?, ?, ?, ?)`,
                [
                    battleId,
                    requirements.bpm_min,
                    requirements.bpm_max,
                    requirements.key_signature,
                    requirements.required_samples,
                    requirements.additional_rules
                ]
            );
        }

        // Add judges if provided
        if (judges && judges.length > 0) {
            const judgeValues = judges.map(judge => [battleId, judge.user_id, judge.weight || 1.0]);
            await db.query(
                'INSERT INTO judges (battle_id, user_id, weight) VALUES ?',
                [judgeValues]
            );
        }

        // Create tournament brackets if it's a tournament battle
        if (tournament_id) {
            await db.query(
                'INSERT INTO tournament_brackets (tournament_id, battle_id, round_number) VALUES (?, ?, ?)',
                [tournament_id, battleId, 1]
            );
        }

        // Update challenge status
        await db.query(
            'UPDATE challenges SET status = ? WHERE id = ?',
            ['active', challenge_id]
        );

        await db.query('COMMIT');

        res.status(201).json({ 
            id: battleId,
            message: 'Battle created successfully'
        });
    } catch (err) {
        await db.query('ROLLBACK');
        console.error('Error creating battle:', err);
        res.status(500).json({ error: 'Internal server error' });
    }
});

// Get battle messages
router.get('/:trackId/messages', async (req, res) => {
    try {
        const [messages] = await db.query(`
            SELECT 
                m.id,
                m.message,
                m.created_at,
                u.username as user_name,
                u.rank as user_rank
            FROM messages m
            JOIN users u ON m.user_id = u.id
            WHERE m.track_id = ?
            ORDER BY m.created_at DESC
            LIMIT 50
        `, [req.params.trackId]);

        res.json(messages);
    } catch (err) {
        console.error('Error fetching messages:', err);
        res.status(500).json({ error: 'Internal server error' });
    }
});

// Add message
router.post('/:trackId/messages', async (req, res) => {
    const { user_id, message } = req.body;
    try {
        const [result] = await db.query(
            'INSERT INTO messages (track_id, user_id, message) VALUES (?, ?, ?)',
            [req.params.trackId, user_id, message]
        );
        res.status(201).json({ 
            id: result.insertId,
            message: 'Message sent successfully'
        });
    } catch (err) {
        console.error('Error sending message:', err);
        res.status(500).json({ error: 'Internal server error' });
    }
});

// Add or update vote
router.post('/:trackId/vote', async (req, res) => {
    const { user_id, ratings } = req.body;
    try {
        await db.query('START TRANSACTION');

        // Add or update main vote
        const [voteResult] = await db.query(
            `INSERT INTO votes (track_id, user_id) 
             VALUES (?, ?) 
             ON DUPLICATE KEY UPDATE created_at = CURRENT_TIMESTAMP`,
            [req.params.trackId, user_id]
        );

        const voteId = voteResult.insertId;

        // Add detailed ratings if provided
        if (ratings) {
            const ratingValues = Object.entries(ratings).map(
                ([category, score]) => [voteId, category, score]
            );
            await db.query(
                `INSERT INTO detailed_votes (vote_id, category, score) 
                 VALUES ? 
                 ON DUPLICATE KEY UPDATE score = VALUES(score)`,
                [ratingValues]
            );
        }

        // Update track's vote count
        await db.query(
            'UPDATE tracks SET votes_count = (SELECT COUNT(*) FROM votes WHERE track_id = ?) WHERE id = ?',
            [req.params.trackId, req.params.trackId]
        );

        await db.query('COMMIT');
        res.json({ message: 'Vote recorded successfully' });
    } catch (err) {
        await db.query('ROLLBACK');
        console.error('Error recording vote:', err);
        res.status(500).json({ error: 'Internal server error' });
    }
});

// Remove vote
router.delete('/:trackId/vote', async (req, res) => {
    const { user_id } = req.body;
    try {
        await db.query('START TRANSACTION');

        // Get vote id first
        const [votes] = await db.query(
            'SELECT id FROM votes WHERE track_id = ? AND user_id = ?',
            [req.params.trackId, user_id]
        );

        if (votes.length > 0) {
            // Delete detailed votes first (due to foreign key)
            await db.query(
                'DELETE FROM detailed_votes WHERE vote_id = ?',
                [votes[0].id]
            );

            // Then delete the main vote
            await db.query(
                'DELETE FROM votes WHERE id = ?',
                [votes[0].id]
            );

            // Update track's vote count
            await db.query(
                'UPDATE tracks SET votes_count = (SELECT COUNT(*) FROM votes WHERE track_id = ?) WHERE id = ?',
                [req.params.trackId, req.params.trackId]
            );
        }

        await db.query('COMMIT');
        res.json({ message: 'Vote removed successfully' });
    } catch (err) {
        await db.query('ROLLBACK');
        console.error('Error removing vote:', err);
        res.status(500).json({ error: 'Internal server error' });
    }
});

// Update battle status
router.patch('/:id/status', async (req, res) => {
    const { status } = req.body