const express = require('express');
const router = express.Router();
const db = require('../db/connection');
const nodeCron = require('node-cron');

// Allowed genres and durations
const allowedGenres = ['Hip Hop', 'Trap', 'Lo-Fi', 'R&B'];
const allowedDurations = ['30 Minutes', '1 Hour', '24 Hours'];

// Create a new challenge
router.post('/', async (req, res) => {
  const { challengerId, challengedId, genre, duration, responseDeadline, message } = req.body;

  // Validate inputs
  if (!challengerId || !challengedId || !genre || !duration || !responseDeadline) {
    return res.status(400).json({ error: 'Missing required fields' });
  }
  if (!allowedGenres.includes(genre)) {
    return res.status(400).json({ error: 'Invalid genre' });
  }
  if (!allowedDurations.includes(duration)) {
    return res.status(400).json({ error: 'Invalid duration' });
  }

  try {
    const query = `
      INSERT INTO challenges (challenger_id, challenged_id, genre, duration, response_deadline, message, status)
      VALUES (?, ?, ?, ?, ?, ?, 'pending')
    `;
    await db.query(query, [challengerId, challengedId, genre, duration, responseDeadline, message]);

    res.status(201).json({ message: 'Challenge created successfully!' });
  } catch (error) {
    console.error('Error creating challenge:', error);
    res.status(500).json({ error: 'Failed to create challenge' });
  }
});

// Fetch all challenges
router.get('/', async (req, res) => {
  try {
    const [results] = await db.query('SELECT * FROM challenges');
    res.status(200).json(results);
  } catch (error) {
    console.error('Error fetching challenges:', error);
    res.status(500).json({ error: 'Failed to fetch challenges' });
  }
});

// Fetch tracks for a specific challenge
router.get('/:id/tracks', async (req, res) => {
  const { id } = req.params;

  try {
    const [results] = await db.query(
      `SELECT t.track_file, u.username
       FROM tracks t
       JOIN users u ON t.participant_id = u.id
       WHERE t.battle_id = ?`,
      [id]
    );

    if (results.length === 0) {
      return res.status(404).json({ error: 'No tracks found for this challenge' });
    }

    res.status(200).json(results);
  } catch (error) {
    console.error('Error fetching tracks:', error);
    res.status(500).json({ error: 'Failed to fetch tracks' });
  }
});

// Update a challenge status
router.put('/:id', async (req, res) => {
  const { id } = req.params;
  const { status } = req.body;

  if (!['pending', 'active', 'canceled', 'completed', 'forfeited'].includes(status)) {
    return res.status(400).json({ error: 'Invalid status' });
  }

  try {
    const query = 'UPDATE challenges SET status = ? WHERE id = ?';
    const [result] = await db.query(query, [status, id]);

    if (result.affectedRows === 0) {
      return res.status(404).json({ error: 'Challenge not found' });
    }

    res.status(200).json({ message: 'Challenge status updated successfully' });
  } catch (error) {
    console.error('Error updating challenge status:', error);
    res.status(500).json({ error: 'Failed to update challenge status' });
  }
});

// Delete a challenge
router.delete('/:id', async (req, res) => {
  const { id } = req.params;

  try {
    const query = 'DELETE FROM challenges WHERE id = ?';
    const [result] = await db.query(query, [id]);

    if (result.affectedRows === 0) {
      return res.status(404).json({ error: 'Challenge not found' });
    }

    res.status(200).json({ message: 'Challenge deleted successfully' });
  } catch (error) {
    console.error('Error deleting challenge:', error);
    res.status(500).json({ error: 'Failed to delete challenge' });
  }
});

// Schedule task to cancel expired challenges
nodeCron.schedule('0 * * * *', async () => {
  try {
    const query = `
      UPDATE challenges
      SET status = 'canceled'
      WHERE status = 'pending' AND response_deadline < NOW()
    `;
    await db.query(query);
    console.log('Checked and canceled expired challenges');
  } catch (error) {
    console.error('Error during scheduled task:', error);
  }
});

module.exports = router;
