const express = require('express');
const cors = require('cors');
const http = require('http');
const WebSocket = require('ws');
const bodyParser = require('body-parser');
const path = require('path');
const db = require('./db/connection');

// Import routes
const challengesRoutes = require('./routes/challenges');
const battlesRoutes = require('./routes/battles');
const leaderboardRoutes = require('./routes/leaderboard');
const votesRoutes = require('./routes/votes');

// Create express app
const app = express();
const server = http.createServer(app);

// Create WebSocket server
const wss = new WebSocket.Server({ server });

// WebSocket connection handling
wss.on('connection', (ws) => {
  console.log('New WebSocket connection');

  // Send initial connection confirmation
  ws.send(JSON.stringify({ type: 'CONNECTION_SUCCESS' }));

  // Handle incoming messages
  ws.on('message', async (message) => {
    try {
      const data = JSON.parse(message);

      switch (data.type) {
        case 'JOIN_BATTLE':
          // Track clients in battle rooms
          ws.battleId = data.battleId;
          broadcastToBattle(data.battleId, {
            type: 'USER_JOINED',
            userId: data.userId
          });
          break;

        case 'LEAVE_BATTLE':
          broadcastToBattle(data.battleId, {
            type: 'USER_LEFT',
            userId: data.userId
          });
          delete ws.battleId;
          break;

        case 'CHAT_MESSAGE':
          // Store message in database
          const [result] = await db.query(
            'INSERT INTO messages (battle_id, user_id, message) VALUES (?, ?, ?)',
            [data.battleId, data.userId, data.message]
          );

          // Broadcast to all clients in the battle
          broadcastToBattle(data.battleId, {
            type: 'NEW_MESSAGE',
            message: {
              id: result.insertId,
              userId: data.userId,
              text: data.message,
              timestamp: new Date()
            }
          });
          break;

        case 'VOTE':
          // Update votes in database
          await db.query(
            'UPDATE tracks SET votes_count = votes_count + 1 WHERE id = ?',
            [data.trackId]
          );

          // Broadcast vote update
          broadcastToBattle(data.battleId, {
            type: 'VOTE_UPDATE',
            trackId: data.trackId,
            totalVotes: data.newTotal
          });
          break;
      }
    } catch (error) {
      console.error('WebSocket message error:', error);
      ws.send(JSON.stringify({ 
        type: 'ERROR', 
        message: 'Failed to process message' 
      }));
    }
  });

  // Handle client disconnection
  ws.on('close', () => {
    if (ws.battleId) {
      broadcastToBattle(ws.battleId, {
        type: 'USER_LEFT',
        userId: ws.userId
      });
    }
  });
});

// Broadcast to all clients in a battle
function broadcastToBattle(battleId, message) {
  wss.clients.forEach((client) => {
    if (client.battleId === battleId && client.readyState === WebSocket.OPEN) {
      client.send(JSON.stringify(message));
    }
  });
}

// Middleware
app.use(cors({
  origin: process.env.NODE_ENV === 'production' 
    ? 'https://your-production-domain.com' 
    : 'http://localhost:3000',
  methods: ['GET', 'POST', 'PUT', 'DELETE', 'PATCH'],
  credentials: true
}));

app.use(bodyParser.json());
app.use(bodyParser.urlencoded({ extended: true }));

// Error handling middleware
app.use((err, req, res, next) => {
  console.error(err.stack);
  res.status(500).json({ 
    error: 'Internal Server Error',
    message: process.env.NODE_ENV === 'development' ? err.message : undefined
  });
});

// Static file serving (for uploaded tracks, etc)
app.use('/uploads', express.static(path.join(__dirname, 'uploads')));

// API Routes
app.use('/api/challenges', challengesRoutes);
app.use('/api/battles', battlesRoutes);
app.use('/api/leaderboard', leaderboardRoutes);
app.use('/api/votes', votesRoutes);

// Test Endpoint
app.get('/', (req, res) => {
  res.send('Welcome to the Tribal Arena Backend!');
});

// Health check endpoint
app.get('/health', (req, res) => {
  res.json({ status: 'ok', timestamp: new Date() });
});

// Database connection check
db.getConnection()
  .then(() => {
    console.log('Database connected successfully');
  })
  .catch((err) => {
    console.error('Database connection failed:', err);
    process.exit(1);
  });

// Start Server
const PORT = process.env.PORT || 5000;
server.listen(PORT, () => {
  console.log(`Server running at http://localhost:${PORT}`);
  console.log(`WebSocket server is running on ws://localhost:${PORT}`);
});

// Graceful shutdown
process.on('SIGTERM', () => {
  console.log('SIGTERM received. Shutting down gracefully...');
  server.close(() => {
    console.log('Server closed');
    db.end(() => {
      console.log('Database connection closed');
      process.exit(0);
    });
  });
});