// websocket.js
const WebSocket = require('ws');
const http = require('http');
const server = http.createServer();
const wss = new WebSocket.Server({ server });

const clients = new Map();

wss.on('connection', (ws) => {
  const clientId = Math.random().toString(36).substring(7);
  clients.set(clientId, ws);

  // Send initial connection success message
  ws.send(JSON.stringify({
    type: 'CONNECTION_SUCCESS',
    clientId
  }));

  ws.on('message', async (message) => {
    try {
      const data = JSON.parse(message);
      
      switch (data.type) {
        case 'JOIN_BATTLE':
          // Handle user joining a battle
          handleJoinBattle(clientId, data.battleId);
          break;

        case 'LEAVE_BATTLE':
          // Handle user leaving a battle
          handleLeaveBattle(clientId, data.battleId);
          break;

        case 'CHAT_MESSAGE':
          // Broadcast chat message to all clients in the battle
          broadcastToBattle(data.battleId, {
            type: 'NEW_CHAT_MESSAGE',
            message: data.message
          });
          break;

        case 'VOTE':
          // Handle vote
          handleVote(data.battleId, data.trackId, data.userId);
          break;
      }
    } catch (error) {
      console.error('WebSocket message error:', error);
      ws.send(JSON.stringify({
        type: 'ERROR',
        message: 'Invalid message format'
      }));
    }
  });

  ws.on('close', () => {
    clients.delete(clientId);
  });
});

// Helper functions
function handleJoinBattle(clientId, battleId) {
  const client = clients.get(clientId);
  if (!client) return;

  // Add client to battle room logic here
  client.battleId = battleId;

  broadcastToBattle(battleId, {
    type: 'USER_JOINED',
    clientId
  });
}

function handleLeaveBattle(clientId, battleId) {
  const client = clients.get(clientId);
  if (!client) return;

  // Remove client from battle room logic here
  delete client.battleId;

  broadcastToBattle(battleId, {
    type: 'USER_LEFT',
    clientId
  });
}

async function handleVote(battleId, trackId, userId) {
  try {
    // Update vote in database
    // await db.query('UPDATE tracks SET votes = votes + 1 WHERE id = ?', [trackId]);

    // Broadcast vote update
    broadcastToBattle(battleId, {
      type: 'VOTE_UPDATE',
      trackId,
      userId
    });
  } catch (error) {
    console.error('Vote handling error:', error);
  }
}

function broadcastToBattle(battleId, message) {
  clients.forEach((client) => {
    if (client.battleId === battleId && client.readyState === WebSocket.OPEN) {
      client.send(JSON.stringify(message));
    }
  });
}

module.exports = { wss, server };