```javascript
    import express from 'express';
    import bodyParser from 'body-parser';
    import { initDatabase } from './db.js';
    import bcrypt from 'bcryptjs';
    import jwt from 'jsonwebtoken';

    const app = express();
    const port = 3000;

    app.use(bodyParser.json());

    // Initialize the database
    const db = await initDatabase();

    // Secret key for JWT (replace with a strong secret in production)
    const JWT_SECRET = 'your-secret-key';

    // Authentication Middleware
    function authenticateToken(req, res, next) {
      const authHeader = req.headers['authorization'];
      const token = authHeader && authHeader.split(' ')[1];

      if (!token) {
        return res.status(401).json({ message: 'Unauthorized' });
      }

      jwt.verify(token, JWT_SECRET, (err, user) => {
        if (err) {
          return res.status(403).json({ message: 'Forbidden' });
        }
        req.user = user;
        next();
      });
    }

    // User Signup
    app.post('/api/signup', async (req, res) => {
      const { username, email, password } = req.body;

      try {
        // Hash the password
        const hashedPassword = await bcrypt.hash(password, 10);

        // Insert the user into the database
        const result = db.run(
          'INSERT INTO users (username, email, password) VALUES (?, ?, ?)',
          [username, email, hashedPassword]
        );

        res.status(201).json({ message: 'User created successfully' });
      } catch (error) {
        if (error.message.includes('UNIQUE constraint failed')) {
          res.status(409).json({ message: 'Username or email already exists' });
        } else {
          console.error('Signup error:', error);
          res.status(500).json({ message: 'Failed to create user' });
        }
      }
    });

    // User Login
    app.post('/api/login', async (req, res) => {
      const { username, password } = req.body;

      try {
        // Retrieve the user from the database
        const user = db.exec(
          'SELECT * FROM users WHERE username = ?',
          [username]
        )[0]?.values[0];

        if (!user) {
          return res.status(401).json({ message: 'Invalid credentials' });
        }

        // Compare the password with the hashed password
        const passwordMatch = await bcrypt.compare(password, user[3]);

        if (passwordMatch) {
          // Generate a JWT
          const token = jwt.sign({ userId: user[0] }, JWT_SECRET, { expiresIn: '1h' });
          res.status(200).json({ token });
        } else {
          res.status(401).json({ message: 'Invalid credentials' });
        }
      } catch (error) {
        console.error('Login error:', error);
        res.status(500).json({ message: 'Failed to log in' });
      }
    });

    // Challenge Management

    // Create a new challenge
    app.post('/api/challenges', authenticateToken, async (req, res) => {
      const { challenged_id, beat_url, duration, name, message } = req.body;
      const challenger_id = req.user.userId;

      try {
        // Validate the challenge data (e.g., challenged user exists)

        // Insert the challenge into the database
        const result = db.run(
          'INSERT INTO challenges (challenger_id, challenged_id, beat_url, duration, name, message) VALUES (?, ?, ?, ?, ?, ?)',
          [challenger_id, challenged_id, beat_url, duration, name, message]
        );

        res.status(201).json({ message: 'Challenge created successfully', challengeId: result.insertId });
      } catch (error) {
        console.error('Challenge creation error:', error);
        res.status(500).json({ message: 'Failed to create challenge' });
      }
    });

    // Get challenges
    app.get('/api/challenges', authenticateToken, async (req, res) => {
      const { status } = req.query;
      const userId = req.user.userId;

      try {
        let query = `
          SELECT * FROM challenges 
          WHERE (challenger_id = ? OR challenged_id = ?)
        `;
        let queryParams = [userId, userId];

        if (status) {
          query += ' AND status = ?';
          queryParams.push(status);
        }

        const challenges = db.exec(query, queryParams)[0]?.values;

        res.status(200).json(challenges);
      } catch (error) {
        console.error('Get challenges error:', error);
        res.status(500).json({ message: 'Failed to retrieve challenges' });
      }
    });

    // Accept a challenge
    app.put('/api/challenges/:id/accept', authenticateToken, async (req, res) => {
      const challengeId = req.params.id;
      const userId = req.user.userId;

      try {
        // Start a transaction
        db.exec('BEGIN');

        // Check if the challenge exists and the user is the challenged user
        const challenge = db.exec(
          'SELECT * FROM challenges WHERE id = ? AND challenged_id = ? AND status = \'pending\'',
          [challengeId, userId]
        )[0]?.values[0];

        if (!challenge) {
          db.exec('ROLLBACK');
          return res.status(404).json({ message: 'Challenge not found or you are not authorized to accept it' });
        }

        // Update the challenge status to 'accepted'
        db.run('UPDATE challenges SET status = \'accepted\' WHERE id = ?', [challengeId]);

        // Create a corresponding battle
        const battleResult = db.run(
          'INSERT INTO battles (challenge_id, start_time, end_time) VALUES (?, datetime(\'now\'), datetime(\'now\', \'+\' || ? || \' hours\'))',
          [challengeId, challenge[4]] // Assuming duration is stored in hours
        );

        // Commit the transaction
        db.exec('COMMIT');

        res.status(200).json({ message: 'Challenge accepted', battleId: battleResult.insertId });
      } catch (error) {
        db.exec('ROLLBACK');
        console.error('Accept challenge error:', error);
        res.status(500).json({ message: 'Failed to accept challenge' });
      }
    });

    // Decline a challenge
    app.put('/api/challenges/:id/decline', authenticateToken, async (req, res) => {
      const challengeId = req.params.id;
      const userId = req.user.userId;

      try {
        // Check if the challenge exists and the user is the challenged user
        const challenge = db.exec(
          'SELECT * FROM challenges WHERE id = ? AND challenged_id = ? AND status = \'pending\'',
          [challengeId, userId]
        )[0]?.values[0];

        if (!challenge) {
          return res.status(404).json({ message: 'Challenge not found or you are not authorized to decline it' });
        }

        // Update the challenge status to 'declined'
        db.run('UPDATE challenges SET status = \'declined\' WHERE id = ?', [challengeId]);

        res.status(200).json({ message: 'Challenge declined' });
      } catch (error) {
        console.error('Decline challenge error:', error);
        res.status(500).json({ message: 'Failed to decline challenge' });
      }
    });

    // Battle Management

    // Get battle details
    app.get('/api/battles/:id', authenticateToken, async (req, res) => {
      const battleId = req.params.id;

      try {
        // Retrieve the battle from the database
        const battle = db.exec(
          'SELECT * FROM battles WHERE id = ?',
          [battleId]
        )[0]?.values[0];

        if (!battle) {
          return res.status(404).json({ message: 'Battle not found' });
        }

        // Get challenge details associated with the battle
        const challenge = db.exec(
          'SELECT * FROM challenges WHERE id = ?',
          [battle[1]] // challenge_id is at index 1 in the battles table
        )[0]?.values[0];

        if (!challenge) {
          return res.status(404).json({ message: 'Challenge associated with the battle not found' });
        }

        // Get challenger and challenged user details
        const challenger = db.exec(
          'SELECT id, username FROM users WHERE id = ?',
          [challenge[1]] // challenger_id is at index 1 in the challenges table
        )[0]?.values[0];

        const challenged = db.exec(
          'SELECT id, username FROM users WHERE id = ?',
          [challenge[2]] // challenged_id is at index 2 in the challenges table
        )[0]?.values[0];

        if (!challenger || !challenged) {
          return res.status(404).json({ message: 'Challenger or challenged user not found' });
        }

        // Get current vote counts for the battle
        const votes = db.exec(
          'SELECT producer_id, COUNT(*) as voteCount FROM votes WHERE battle_id = ? GROUP BY producer_id',
          [battleId]
        );

        const voteCounts = {};
        if (votes[0] && votes[0].values) {
          votes[0].values.forEach(vote => {
            voteCounts[vote[0]] = vote[1];
          });
        }

        // Construct the battle details object
        const battleDetails = {
          id: battle[0],
          challenge: {
            id: challenge[0],
            name: challenge[5],
            message: challenge[6],
            beat_url: challenge[3],
            duration: challenge[4],
            timestamp: challenge[8],
          },
          participants: {
            challenger: {
              id: challenger[0],
              username: challenger[1],
              votes: voteCounts[challenger[0]] || 0
            },
            challenged: {
              id: challenged[0],
              username: challenged[1],
              votes: voteCounts[challenged[0]] || 0
            }
          },
          start_time: battle[2],
          end_time: battle[3],
        };

        res.status(200).json(battleDetails);
      } catch (error) {
        console.error('Get battle details error:', error);
        res.status(500).json({ message: 'Failed to retrieve battle details' });
      }
    });

    // End a battle
    app.put('/api/battles/:id/end', authenticateToken, async (req, res) => {
      const battleId = req.params.id;

      try {
        // Retrieve the battle from the database
        const battle = db.exec(
          'SELECT * FROM battles WHERE id = ?',
          [battleId]
        )[0]?.values[0];

        if (!battle) {
          return res.status(404).json({ message: 'Battle not found' });
        }

        // Check if the battle has already ended
        if (battle[3] && new Date(battle[3]) < new Date()) {
          return res.status(400).json({ message: 'Battle has already ended' });
        }

        // Calculate the winner based on votes
        const votes = db.exec(
          'SELECT producer_id, COUNT(*) as voteCount FROM votes WHERE battle_id = ? GROUP BY producer_id',
          [battleId]
        );

        let winnerId = null;
        if (votes[0] && votes[0].values) {
          const voteCounts = votes[0].values;
          if (voteCounts.length > 0) {
            winnerId = voteCounts.reduce((prev, current) => (prev[1] > current[1]) ? prev : current)[0];
          }
        }

        // Update the battle end time
        db.run('UPDATE battles SET end_time = datetime(\'now\') WHERE id = ?', [battleId]);

        // Update the leaderboard (simplified for this example)
        if (winnerId) {
          db.run('INSERT OR IGNORE INTO leaderboard (user_id) VALUES (?)', [winnerId]);
          db.run('UPDATE leaderboard SET total_wins = total_wins + 1, battles_participated = battles_participated + 1, win_rate = (total_wins * 1.0 / battles_participated) WHERE user_id = ?', [winnerId]);
        }

        res.status(200).json({ message: 'Battle ended', winnerId });
      } catch (error) {
        console.error('End battle error:', error);
        res.status(500).json({ message: 'Failed to end battle' });
      }
    });

    // Voting Endpoints

    // Cast a vote for a producer in a battle
    app.post('/api/battles/:id/vote', authenticateToken, async (req, res) => {
      const battleId = req.params.id;
      const userId = req.user.userId;
      const { producer_id } = req.body; // The ID of the producer being voted for

      try {
        // Check if the battle exists and is still active
        const battle = db.exec(
          'SELECT * FROM battles WHERE id = ? AND datetime(\'now\') BETWEEN start_time AND end_time',
          [battleId]
        )[0]?.values[0];

        if (!battle) {
          return res.status(404).json({ message: 'Battle not found or has ended' });
        }

        // Check if the user has already voted in this battle
        const existingVote = db.exec(
          'SELECT * FROM votes WHERE battle_id = ? AND user_id = ?',
          [battleId, userId]
        )[0]?.values[0];

        if (existingVote) {
          return res.status(400).json({ message: 'User has already voted in this battle' });
        }

        // Insert the vote into the database
        db.run(
          'INSERT INTO votes (battle_id, user_id, producer_id) VALUES (?, ?, ?)',
          [battleId, userId, producer_id]
        );

        res.status(201).json({ message: 'Vote recorded successfully' });
      } catch (error) {
        console.error('Vote error:', error);
        res.status(500).json({ message: 'Failed to record vote' });
      }
    });

    // Get vote counts for a battle
    app.get('/api/battles/:id/votes', authenticateToken, async (req, res) => {
      const battleId = req.params.id;

      try {
        // Retrieve the vote counts from the database
        const votes = db.exec(
          'SELECT producer_id, COUNT(*) as voteCount FROM votes WHERE battle_id = ? GROUP BY producer_id',
          [battleId]
        );

        const voteCounts = {};
        if (votes[0] && votes[0].values) {
          votes[0].values.forEach(vote => {
            voteCounts[vote[0]] = vote[1];
          });
        }

        res.status(200).json(voteCounts);
      } catch (error) {
        console.error('Get votes error:', error);
        res.status(500).json({ message: 'Failed to retrieve vote counts' });
      }
    });

    // Leaderboard Endpoint

    // Get the leaderboard
    app.get('/api/leaderboard', authenticateToken, async (req, res) => {
      try {
        // Retrieve the leaderboard from the database
        const leaderboard = db.exec(
          'SELECT u.username, l.total_wins, l.battles_participated, l.win_rate FROM leaderboard l JOIN users u ON l.user_id = u.id ORDER BY l.total_wins DESC, l.win_rate DESC'
        )[0]?.values;

        if (!leaderboard) {
          return res.status(404).json({ message: 'Leaderboard not found' });
        }

        // Map the leaderboard data to a more readable format
        const formattedLeaderboard = leaderboard.map(row => ({
          username: row[0],
          total_wins: row[1],
          battles_participated: row[2],
          win_rate: row[3]
        }));

        res.status(200).json(formattedLeaderboard);
      } catch (error) {
        console.error('Get leaderboard error:', error);
        res.status(500).json({ message: 'Failed to retrieve leaderboard' });
      }
    });

    // User Stats Endpoint

    // Get user stats
    app.get('/api/users/:id/stats', authenticateToken, async (req, res) => {
      const userId = req.params.id;

      try {
        // Retrieve the user stats from the database
        const stats = db.exec(
          'SELECT total_wins, battles_participated, win_rate FROM leaderboard WHERE user_id = ?',
          [userId]
        )[0]?.values[0];

        if (!stats) {
          return res.status(404).json({ message: 'User stats not found' });
        }

        // Retrieve the user's badges
        const badges = db.exec(
          'SELECT b.name, b.description, b.icon_url FROM user_badges ub JOIN badges b ON ub.badge_id = b.id WHERE ub.user_id = ?',
          [userId]
        )[0]?.values;

        // Map the badges data to a more readable format
        const formattedBadges = badges ? badges.map(badge => ({
          name: badge[0],
          description: badge[1],
          icon_url: badge[2]
        })) : [];

        res.status(200).json({
          total_wins: stats[0],
          battles_participated: stats[1],
          win_rate: stats[2],
          badges: formattedBadges
        });
      } catch (error) {
        console.error('Get user stats error:', error);
        res.status(500).json({ message: 'Failed to retrieve user stats' });
      }
    });

    // Theme Management Endpoints

    // Create a new theme (admin only)
    app.post('/api/themes', authenticateToken, async (req, res) => {
      // In a real application, you would check if the user is an admin here
      const { name, description, start_date, end_date } = req.body;

      try {
        // Insert the theme into the database
        const result = db.run(
          'INSERT INTO themes (name, description, start_date, end_date) VALUES (?, ?, ?, ?)',
          [name, description, start_date, end_date]
        );

        res.status(201).json({ message: 'Theme created successfully', themeId: result.insertId });
      } catch (error) {
        console.error('Theme creation error:', error);
        res.status(500).json({ message: 'Failed to create theme' });
      }
    });

    // Get a list of themes
    app.get('/api/themes', async (req, res) => {
      try {
        // Retrieve the themes from the database
        const themes = db.exec(
          'SELECT * FROM themes'
        )[0]?.values;

        // Map the themes data to a more readable format
        const formattedThemes = themes.map(theme => ({
          id: theme[0],
          name: theme[1],
          description: theme[2],
          start_date: theme[3],
          end_date: theme[4]
        }));

        res.status(200).json(formattedThemes);
      } catch (error) {
        console.error('Get themes error:', error);
        res.status(500).json({ message: 'Failed to retrieve themes' });
      }
    });

    // Update a theme (admin only)
    app.put('/api/themes/:id', authenticateToken, async (req, res) => {
      // In a real application, you would check if the user is an admin here
      const themeId = req.params.id;
      const { name, description, start_date, end_date } = req.body;

      try {
        // Update the theme in the database
        db.run(
          'UPDATE themes SET name = ?, description = ?, start_date = ?, end_date = ? WHERE id = ?',
          [name, description, start_date, end_date, themeId]
        );

        res.status(200).json({ message: 'Theme updated successfully' });
      } catch (error) {
        console.error('Theme update error:', error);
        res.status(500).json({ message: 'Failed to update theme' });
      }
    });

    // Delete a theme (admin only)
    app.delete('/api/themes/:id', authenticateToken, async (req, res) => {
      // In a real application, you would check if the user is an admin here
      const themeId = req.params.id;

      try {
        // Delete the theme from the database
        db.run('DELETE FROM themes WHERE id = ?', [themeId]);

        res.status(200).json({ message: 'Theme deleted successfully' });
      } catch (error) {
        console.error('Theme deletion error:', error);
        res.status(500).json({ message: 'Failed to delete theme' });
      }
    });

    // Badge Management Endpoints

    // Create a new badge (admin only)
    app.post('/api/badges', authenticateToken, async (req, res) => {
      // In a real application, you would check if the user is an admin here
      const { name, description, icon_url } = req.body;

      try {
        // Insert the badge into the database
        const result = db.run(
          'INSERT INTO badges (name, description, icon_url) VALUES (?, ?, ?)',
          [name, description, icon_url]
        );

        res.status(201).json({ message: 'Badge created successfully', badgeId: result.insertId });
      } catch (error) {
        console.error('Badge creation error:', error);
        res.status(500).json({ message: 'Failed to create badge' });
      }
    });

    // Get a list of badges
    app.get('/api/badges', async (req, res) => {
      try {
        // Retrieve the badges from the database
        const badges = db.exec(
          'SELECT * FROM badges'
        )[0]?.values;

        // Map the badges data to a more readable format
        const formattedBadges = badges.map(badge => ({
          id: badge[0],
          name: badge[1],
          description: badge[2],
          icon_url: badge[3]
        }));

        res.status(200).json(formattedBadges);
      } catch (error) {
        console.error('Get badges error:', error);
        res.status(500).json({ message: 'Failed to retrieve badges' });
      }
    });

    // Update a badge (admin only)
    app.put('/api/badges/:id', authenticateToken, async (req, res) => {
      // In a real application, you would check if the user is an admin here
      const badgeId = req.params.id;
      const { name, description, icon_url } = req.body;

      try {
        // Update the badge in the database
        db.run(
          'UPDATE badges SET name = ?, description = ?, icon_url = ? WHERE id = ?',
          [name, description, icon_url, badgeId]
        );

        res.status(200).json({ message: 'Badge updated successfully' });
      } catch (error) {
        console.error('Badge update error:', error);
        res.status(500).json({ message: 'Failed to update badge' });
      }
    });

    // Delete a badge (admin only)
    app.delete('/api/badges/:id', authenticateToken, async (req, res) => {
      // In a real application, you would check if the user is an admin here
      const badgeId = req.params.id;

      try {
        // Delete the badge from the database
        db.run('DELETE FROM badges WHERE id = ?', [badgeId]);

        res.status(200).json({ message: 'Badge deleted successfully' });
      } catch (error) {
        console.error('Badge deletion error:', error);
        res.status(500).json({ message: 'Failed to delete badge' });
      }
    });

    // Award a badge to a user
    app.post('/api/users/:id/badges', authenticateToken, async (req, res) => {
      const userId = req.params.id;
      const { badgeId } = req.body;

      try {
        // Insert the user badge into the database
        const result = db.run(
          'INSERT INTO user_badges (user_id, badge_id) VALUES (?, ?)',
          [userId, badgeId]
        );

        res.status(201).json({ message: 'Badge awarded successfully', userBadgeId: result.insertId });
      } catch (error) {
        console.error('Award badge error:', error);
        res.status(500).json({ message: 'Failed to award badge' });
      }
    });

    // Admin Features Endpoints

    // Get a list of battles for moderation (admin only)
    app.get('/api/battles/moderation', authenticateToken, async (req, res) => {
      // In a real application, you would check if the user is an admin here

      try {
        // Retrieve the battles from the database
        const battles = db.exec(
          'SELECT * FROM battles'
        )[0]?.values;

        // Map the battles data to a more readable format
        const formattedBattles = battles.map(battle => ({
          id: battle[0],
          challenge_id: battle[1],
          start_time: battle[2],
          end_time: battle[3]
        }));

        res.status(200).json(formattedBattles);
      } catch (error) {
        console.error('Get battles for moderation error:', error);
        res.status(500).json({ message: 'Failed to retrieve battles for moderation' });
      }
    });

    // Moderate a battle (approve/reject) (admin only)
    app.put('/api/battles/:id/moderate', authenticateToken, async (req, res) => {
      // In a real application, you would check if the user is an admin here
      const battleId = req.params.id;
      const { status } = req.body; // 'approved' or 'rejected'

      try {
        // Update the battle status in the database
        // In a real application, you might have a separate 'status' field for moderation
        db.run('UPDATE battles SET end_time = ? WHERE id = ?', [status === 'approved' ? null : new Date(), battleId]);

        res.status(200).json({ message: `Battle ${status} successfully` });
      } catch (error) {
        console.error('Moderate battle error:', error);
        res.status(500).json({ message: 'Failed to moderate battle' });
      }
    });

    // Feature a battle (admin only)
    app.post('/api/battles/featured', authenticateToken, async (req, res) => {
      // In a real application, you would check if the user is an admin here
      const { battleId } = req.body;
    
      try {
        // Add the battle to a featured battles table or update a field in the battles table
        // This is a simplified example, you might need a separate table to manage featured battles
        db.run('UPDATE battles SET featured = 1 WHERE id = ?', [battleId]);
    
        res.status(200).json({ message: 'Battle featured successfully' });
      } catch (error) {
        console.error('Feature battle error:', error);
        res.status(500).json({ message: 'Failed to feature battle' });
      }
    });
    
    // Get a list of wallpapers (admin only)
    app.get('/api/wallpapers', authenticateToken, async (req, res) => {
      // In a real application, you would check if the user is an admin here
    
      try {
        // Retrieve the wallpapers from the database
        const wallpapers = db.exec(
          'SELECT * FROM wallpapers'
        )[0]?.values;
    
        // Map the wallpapers data to a more readable format
        const formattedWallpapers = wallpapers.map(wallpaper => ({
          id: wallpaper[0],
          url: wallpaper[1],
          name: wallpaper[2],
          active: wallpaper[3]
        }));
    
        res.status(200).json(formattedWallpapers);
      } catch (error) {
        console.error('Get wallpapers error:', error);
        res.status(500).json({ message: 'Failed to retrieve wallpapers' });
      }
    });
    
    // Upload a new wallpaper (admin only)
    app.post('/api/wallpapers', authenticateToken, async (req, res) => {
      // In a real application, you would check if the user is an admin here
      const { url, name } = req.body;
    
      try {
        // Insert the wallpaper into the database
        const result = db.run(
          'INSERT INTO wallpapers (url, name, active) VALUES (?, ?, 0)',
          [url, name]
        );
    
        res.status(201).json({ message: 'Wallpaper uploaded successfully', wallpaperId: result.insertId });
      } catch (error) {
        console.error('Upload wallpaper error:', error);
        res.status(500).json({ message: 'Failed to upload wallpaper' });
      }
    });
    
    // Update a wallpaper (admin only)
    app.put('/api/wallpapers/:id', authenticateToken, async (req, res) => {
      // In a real application, you would check if the user is an admin here
      const wallpaperId = req.params.id;
      const { url, name, active } = req.body;
    
      try {
        // Update the wallpaper in the database
        db.run(
          'UPDATE wallpapers SET url = ?, name = ?, active = ? WHERE id = ?',
          [url, name, active, wallpaperId]
        );
    
        res.status(200).json({ message: 'Wallpaper updated successfully' });
      } catch (error) {
        console.error('Update wallpaper error:', error);
        res.status(500).json({ message: 'Failed to update wallpaper' });
      }
    });
    
    // Delete a wallpaper (admin only)
    app.delete('/api/wallpapers/:id', authenticateToken, async (req, res) => {
      // In a real application, you would check if the user is an admin here
      const wallpaperId = req.params.id;
    
      try {
        // Delete the wallpaper from the database
        db.run('DELETE FROM wallpapers WHERE id = ?', [wallpaperId]);
    
        res.status(200).json({ message: 'Wallpaper deleted successfully' });
      } catch (error) {
        console.error('Delete wallpaper error:', error);
        res.status(500).json({ message: 'Failed to delete wallpaper' });
      }
    });

    const server = app.listen(port, () => {
      console.log(`Server running on port ${port}`);
    });

    // Handle server shutdown gracefully
    process.on('SIGINT', () => {
      console.log('Shutting down server...');
      server.close(() => {
        console.log('Server stopped.');
        db.close();
        process.exit(0);
      });
    });

    export default app;
