import React, { useState, useEffect, useCallback } from 'react';
import BattleCard from './BattleCard';
import LeaderboardPanel from './LeaderboardPanel';
import StartBattleModal from './StartBattleModal';
import JoinBattleModal from './JoinBattleModal';
import { FiFilter } from 'react-icons/fi';
import './BattleArena.css';

function BattleArena() {
  const [activeBattles, setActiveBattles] = useState([]);
  const [selectedGenre, setSelectedGenre] = useState('All');
  const [selectedBattleType, setBattleType] = useState('All');
  const [isStartModalOpen, setIsStartModalOpen] = useState(false);
  const [isJoinModalOpen, setIsJoinModalOpen] = useState(false);
  const [isLoading, setIsLoading] = useState(true);
  const [error, setError] = useState(null);
  const [battleTypes, setBattleTypes] = useState([]);
  const [filters, setFilters] = useState({
    genre: 'All',
    type: 'All',
    status: 'All',
    prizePool: 'All'
  });

  const genres = ['All', 'Hip Hop', 'Trap', 'Lo-Fi', 'R&B'];

  // Mock battle types fetch
  const fetchBattleTypes = useCallback(async () => {
    try {
      // Simulated API response
      const mockTypes = ['All', 'Direct Challenge', 'Community Battle', 'Tournament'];
      setBattleTypes(mockTypes);
    } catch (error) {
      console.error('Error fetching battle types:', error);
      setError('Failed to load battle types');
    }
  }, []);

  // Mock battles fetch
  const fetchBattles = useCallback(async () => {
    try {
      setIsLoading(true);
      // Simulated API response
      const mockBattles = [
        {
          id: 1,
          type: 'Direct Challenge',
          status: 'LIVE',
          producer1: {
            id: 1,
            name: 'Producer X',
            rank: 'Gold',
            xp: 1500,
            track: '/mock-track-1.mp3',
            votes: 150
          },
          producer2: {
            id: 2,
            name: 'Producer Y',
            rank: 'Silver',
            xp: 1200,
            track: '/mock-track-2.mp3',
            votes: 120
          },
          genre: 'Hip Hop',
          roundInfo: 'Quarter Finals',
          viewers: 234,
          timeLeft: '15:00',
          prizePool: 1000
        },
        // Add more mock battles as needed
      ];
      
      setActiveBattles(mockBattles);
      setError(null);
    } catch (error) {
      console.error('Error fetching battles:', error);
      setError('Failed to load battles');
    } finally {
      setIsLoading(false);
    }
  }, []);

  useEffect(() => {
    fetchBattleTypes();
    fetchBattles();

    // Set up polling for updates instead of WebSocket
    const pollInterval = setInterval(fetchBattles, 60000);

    return () => {
      clearInterval(pollInterval);
    };
  }, [fetchBattles, fetchBattleTypes]);

  const handleStartBattle = async (battleData) => {
    try {
      // Mock API call
      const newBattle = {
        id: Date.now(),
        ...battleData,
        status: 'pending',
        type: 'Direct Challenge',
      };
      
      setActiveBattles(prevBattles => [...prevBattles, newBattle]);
      setIsStartModalOpen(false);
    } catch (error) {
      console.error('Error creating battle:', error);
    }
  };

  const handleFilterChange = (filterType, value) => {
    setFilters(prev => ({
      ...prev,
      [filterType]: value
    }));
  };

  const applyFilters = (battles) => {
    return battles.filter(battle => {
      const genreMatch = filters.genre === 'All' || battle.genre === filters.genre;
      const typeMatch = filters.type === 'All' || battle.type === filters.type;
      const statusMatch = filters.status === 'All' || battle.status === filters.status;
      const prizeMatch = filters.prizePool === 'All' || 
        (filters.prizePool === 'Prize' ? battle.prizePool > 0 : true);
      
      return genreMatch && typeMatch && statusMatch && prizeMatch;
    });
  };

  const filteredBattles = applyFilters(activeBattles);

  if (error) {
    return (
      <div className="battle-arena">
        <div className="error-message">
          {error}
          <button onClick={fetchBattles}>Retry</button>
        </div>
      </div>
    );
  }

  return (
    <div className="battle-arena">
      <StartBattleModal
        isOpen={isStartModalOpen}
        onClose={() => setIsStartModalOpen(false)}
        onSubmit={handleStartBattle}
        battleTypes={battleTypes}
      />

      <JoinBattleModal
        isOpen={isJoinModalOpen}
        onClose={() => setIsJoinModalOpen(false)}
      />

      <div className="battle-header">
        <h1>Tribal Arena</h1>
        <div className="battle-actions">
          <button className="btn-primary" onClick={() => setIsStartModalOpen(true)}>
            Start Battle
          </button>
          <button className="btn-secondary" onClick={() => setIsJoinModalOpen(true)}>
            Join Battle
          </button>
        </div>
      </div>

      <div className="battle-grid">
        <div className="battle-main">
          <div className="battle-types">
            <h2>Active Battles</h2>
            <div className="battle-filters">
              <div className="filter-group">
                <select
                  className="genre-dropdown"
                  value={filters.genre}
                  onChange={(e) => handleFilterChange('genre', e.target.value)}
                >
                  {genres.map((genre) => (
                    <option key={genre} value={genre}>{genre}</option>
                  ))}
                </select>

                <select
                  className="type-dropdown"
                  value={filters.type}
                  onChange={(e) => handleFilterChange('type', e.target.value)}
                >
                  {battleTypes.map((type) => (
                    <option key={type} value={type}>{type}</option>
                  ))}
                </select>

                <select
                  className="status-dropdown"
                  value={filters.status}
                  onChange={(e) => handleFilterChange('status', e.target.value)}
                >
                  <option value="All">All Status</option>
                  <option value="LIVE">Live</option>
                  <option value="VOTING">Voting</option>
                </select>

                <select
                  className="prize-dropdown"
                  value={filters.prizePool}
                  onChange={(e) => handleFilterChange('prizePool', e.target.value)}
                >
                  <option value="All">All Battles</option>
                  <option value="Prize">Prize Pool</option>
                </select>
              </div>

              <button className="filter-button">
                <FiFilter />
                Filters
              </button>
            </div>
          </div>

          <div className="active-battles">
            {isLoading ? (
              <div className="loading-spinner">Loading battles...</div>
            ) : filteredBattles.length > 0 ? (
              filteredBattles.map((battle) => (
                <BattleCard 
                  key={battle.id} 
                  battle={battle}
                  onVote={async (trackId, userId) => {
                    // Mock vote handling
                    const updatedBattles = activeBattles.map(b => {
                      if (b.id === battle.id) {
                        return {
                          ...b,
                          producer1: trackId === b.producer1.id ? 
                            { ...b.producer1, votes: b.producer1.votes + 1 } : 
                            b.producer1,
                          producer2: trackId === b.producer2.id ? 
                            { ...b.producer2, votes: b.producer2.votes + 1 } : 
                            b.producer2
                        };
                      }
                      return b;
                    });
                    setActiveBattles(updatedBattles);
                  }}
                />
              ))
            ) : (
              <div className="no-battles">
                <p>No battles found matching your filters.</p>
                <button onClick={() => setFilters({
                  genre: 'All',
                  type: 'All',
                  status: 'All',
                  prizePool: 'All'
                })}>Clear Filters</button>
              </div>
            )}
          </div>
        </div>

        <div className="battle-sidebar">
          <LeaderboardPanel />
        </div>
      </div>
    </div>
  );
}

export default BattleArena;
