import React, { useState, useRef, useEffect } from 'react';
import { FaFire, FaPlay, FaPause, FaVolumeUp, FaVolumeMute, FaTrophy, FaUserCircle } from 'react-icons/fa';
import { BsPeople, BsStars } from 'react-icons/bs';
import { BiTime, BiMedal } from 'react-icons/bi';
import { IoMdSend } from 'react-icons/io';
import { useNavigate } from 'react-router-dom';
import './BattleCard.css';

const AudioPlayer = ({ src }) => {
  const [isPlaying, setIsPlaying] = useState(false);
  const [currentTime, setCurrentTime] = useState(0);
  const audioRef = useRef(null);

  const togglePlay = () => {
    if (isPlaying) {
      audioRef.current?.pause();
    } else {
      audioRef.current?.play();
    }
    setIsPlaying(!isPlaying);
  };

  return (
    <div className="audio-player">
      <audio ref={audioRef} src={src} onTimeUpdate={() => setCurrentTime(audioRef.current?.currentTime || 0)} />
      <button className="play-button" onClick={togglePlay}>
        {isPlaying ? <FaPause /> : <FaPlay />}
      </button>
      <div className="progress-bar">
        <div className="progress" style={{ width: `${(currentTime / (audioRef.current?.duration || 1)) * 100}%` }}></div>
      </div>
    </div>
  );
};

const ProducerStats = ({ producer }) => (
  <div className="producer-stats">
    <div className="producer-rank">
      <BiMedal className={`rank-icon ${producer.rank?.toLowerCase()}`} />
      <span>{producer.rank || 'Unranked'}</span>
    </div>
    <div className="producer-xp">
      <BsStars />
      <span>{producer.xp || 0} XP</span>
    </div>
    <div className="producer-wins">
      <FaTrophy />
      <span>{producer.wins || 0} Wins</span>
    </div>
  </div>
);

function BattleCard({ battle, onVote }) {
  const navigate = useNavigate();
  const [messages1, setMessages1] = useState([]);
  const [messages2, setMessages2] = useState([]);
  const [newMessage1, setNewMessage1] = useState('');
  const [newMessage2, setNewMessage2] = useState('');
  const [isVoting, setIsVoting] = useState(false);
  const chatRef1 = useRef(null);
  const chatRef2 = useRef(null);

  useEffect(() => {
    // Mock initial messages
    const mockMessages = [
      { user: 'User1', text: 'Great beat!', timestamp: new Date().toISOString() },
      { user: 'User2', text: 'Fire!', timestamp: new Date().toISOString() }
    ];
    
    setMessages1(mockMessages);
    setMessages2(mockMessages);

    const messageInterval = setInterval(() => {
      const newMessage = {
        user: `User${Math.floor(Math.random() * 100)}`,
        text: 'New message!',
        timestamp: new Date().toISOString()
      };

      setMessages1(prev => [...prev, newMessage]);
      setMessages2(prev => [...prev, newMessage]);
    }, 30000);

    return () => clearInterval(messageInterval);
  }, [battle]);

  const handleJoinBattle = () => {
    navigate(`/battle/${battle.id}`);
  };

  const handleSendMessage = async (trackId, message, setMessages, setNewMessage) => {
    if (!message.trim()) return;

    const mockMessage = {
      user: 'Current User',
      text: message.trim(),
      timestamp: new Date().toISOString()
    };

    setMessages(prev => [...prev, mockMessage]);
    setNewMessage('');
  };

  const handleVote = async (trackId) => {
    if (isVoting) return;
    setIsVoting(true);
    try {
      await onVote?.(trackId, 1); // Replace 1 with actual user ID
    } catch (error) {
      console.error('Error voting:', error);
    } finally {
      setIsVoting(false);
    }
  };

  const handleSubmitTrack = () => {
    // Implement track submission logic
    console.log('Submitting track for battle:', battle.id);
  };

  const renderProducer = (producer, messages, newMessage, setNewMessage, setMessages, chatRef) => (
    <div className="producer">
      <div className="producer-header">
        <h3>{producer?.name || 'Unknown Producer'}</h3>
        <ProducerStats producer={producer} />
      </div>

      <AudioPlayer src={producer?.track || '/default-track.mp3'} />

      <div className="fire-votes" onClick={() => handleVote(producer?.trackId)}>
        {Array(5).fill().map((_, i) => (
          <FaFire
            key={i}
            className={`${i < Math.floor(producer?.votes / 50) ? 'active' : ''} ${isVoting ? 'voting' : ''}`}
          />
        ))}
        <span className="vote-count">{producer?.votes || 0}</span>
      </div>

      <div className="chat-section">
        <div className="chat-messages" ref={chatRef}>
          {messages?.map((msg, index) => (
            <div key={index} className="message">
              <div className="message-header">
                <FaUserCircle className="user-avatar" />
                <span className="message-user">{msg.user}</span>
                <span className="message-time">{msg.timestamp}</span>
              </div>
              <div className="message-content">
                <span className="message-text">{msg.text}</span>
              </div>
            </div>
          ))}
        </div>
        <div className="chat-input">
          <input
            type="text"
            value={newMessage}
            onChange={(e) => setNewMessage(e.target.value)}
            placeholder="Type a message..."
            onKeyPress={(e) => {
              if (e.key === 'Enter') {
                handleSendMessage(producer?.trackId, newMessage, setMessages, setNewMessage);
              }
            }}
          />
          <button
            onClick={() => handleSendMessage(producer?.trackId, newMessage, setMessages, setNewMessage)}
            className="send-button"
          >
            <IoMdSend />
          </button>
        </div>
      </div>
    </div>
  );

  const renderCommunityBattle = () => (
    <div className="community-battle">
      <div className="participants-grid">
        {battle.participants?.map((participant, index) => (
          <div key={index} className="participant-card">
            {renderProducer(participant, [], '', () => {}, () => {}, null)}
            <div className="participant-rank">
              #{index + 1}
            </div>
          </div>
        ))}
      </div>
      <div className="submission-info">
        <div className="time-remaining">
          <BiTime /> Submissions close in: {battle.submissionTimeLeft}
        </div>
        <button 
          className="btn-submit"
          onClick={handleSubmitTrack}
          disabled={battle.status === 'VOTING' || battle.status === 'COMPLETED'}
        >
          Submit Track
        </button>
      </div>
    </div>
  );

  const renderTournamentBattle = () => (
    <div className="tournament-battle">
      <div className="tournament-bracket">
        <div className="tournament-round">
          <h4>Round {battle.currentRound} of {battle.totalRounds}</h4>
          <div className="bracket-matches">
            {battle.currentMatches?.map((match, index) => (
              <div key={index} className="bracket-match">
                <div className="bracket-player">
                  {match.producer1.name}
                  {match.producer1.track && (
                    <div className="mini-player">
                      {renderProducer(match.producer1, [], '', () => {}, () => {}, null)}
                    </div>
                  )}
                </div>
                <div className="bracket-vs">VS</div>
                <div className="bracket-player">
                  {match.producer2.name}
                  {match.producer2.track && (
                    <div className="mini-player">
                      {renderProducer(match.producer2, [], '', () => {}, () => {}, null)}
                    </div>
                  )}
                </div>
              </div>
            ))}
          </div>
        </div>
        {battle.prizePool > 0 && (
          <div className="tournament-prize">
            <FaTrophy />
            <span>Prize Pool: ${battle.prizePool}</span>
          </div>
        )}
      </div>
    </div>
  );

  return (
    <div className="battle-card">
      <div className="battle-card-header">
        <div className="battle-status-group">
          <span className={`battle-status ${battle.status?.toLowerCase() || ''}`}>
            {battle.status || 'Unknown'}
          </span>
          {battle.tournament && (
            <span className="tournament-badge">
              <FaTrophy /> {battle.tournament.name}
            </span>
          )}
        </div>
        <div className="battle-info-group">
          <span className="battle-type">{battle.type || 'N/A'}</span>
          <span className="battle-genre">{battle.genre || 'Unspecified Genre'}</span>
          {battle.prizePool > 0 && (
            <span className="prize-pool">
              Prize: ${battle.prizePool}
            </span>
          )}
        </div>
      </div>

      <div className="battle-content">
        {battle.type === 'Direct Challenge' ? (
          <div className="versus-battle">
            {renderProducer(battle.producer1, messages1, newMessage1, setNewMessage1, setMessages1, chatRef1)}
            <div className="vs-container">
              <span className="vs">VS</span>
              {battle.roundInfo && (
                <span className="round-badge">{battle.roundInfo}</span>
              )}
            </div>
            {renderProducer(battle.producer2, messages2, newMessage2, setNewMessage2, setMessages2, chatRef2)}
          </div>
        ) : battle.type === 'Community Battle' ? (
          renderCommunityBattle()
        ) : battle.type === 'Tournament' ? (
          renderTournamentBattle()
        ) : (
          <p>No data available for this battle type.</p>
        )}
      </div>

      <div className="battle-footer">
        <div className="battle-info">
          <span className="round-info">{battle.roundInfo || 'No Round Info'}</span>
          <span className="viewers">
            <BsPeople /> {battle.viewers || 0}
          </span>
          <span className="time-left">
            <BiTime /> {battle.timeLeft || '00:00'}
          </span>
        </div>
        <div className="battle-actions">
          <button 
            className="btn-join" 
            onClick={handleJoinBattle}
            disabled={battle.status === 'COMPLETED'}
          >
            Join Battle
          </button>
          <button 
            className="btn-challenge"
            disabled={battle.status === 'COMPLETED'}
          >
            Challenge
          </button>
        </div>
      </div>
    </div>
  );
}

export default BattleCard;