import React, { useState } from 'react'
import { FaPlus, FaTimes } from 'react-icons/fa'
import './EffectsPanel.css'

function EffectsPanel({ tracks, onUpdateTrack }) {
  const [availableEffects] = useState(['Reverb', 'Delay', 'Chorus', 'Flanger', 'Phaser', 'Compressor'])

  const handleAddEffect = (trackId, effect) => {
    const track = tracks.find(t => t.id === trackId)
    const effects = track.effects || []
    onUpdateTrack(trackId, {
      effects: [...effects, {
        type: effect,
        id: Date.now(),
        settings: getDefaultSettings(effect)
      }]
    })
  }

  const handleRemoveEffect = (trackId, effectId) => {
    const track = tracks.find(t => t.id === trackId)
    onUpdateTrack(trackId, {
      effects: track.effects.filter(e => e.id !== effectId)
    })
  }

  const handleEffectSettingChange = (trackId, effectId, setting, value) => {
    const track = tracks.find(t => t.id === trackId)
    onUpdateTrack(trackId, {
      effects: track.effects.map(effect => 
        effect.id === effectId 
          ? { ...effect, settings: { ...effect.settings, [setting]: value } }
          : effect
      )
    })
  }

  const getDefaultSettings = (effectType) => {
    switch (effectType) {
      case 'Reverb':
        return { mix: 0.3, decay: 2.0, preDelay: 0.0 }
      case 'Delay':
        return { mix: 0.3, time: 0.5, feedback: 0.4 }
      case 'Chorus':
        return { mix: 0.5, rate: 1.5, depth: 0.7 }
      case 'Flanger':
        return { mix: 0.5, rate: 0.5, depth: 0.5, feedback: 0.5 }
      case 'Phaser':
        return { mix: 0.5, rate: 0.5, depth: 0.5, feedback: 0.5 }
      case 'Compressor':
        return { threshold: -24, ratio: 4, attack: 0.003, release: 0.25 }
      default:
        return {}
    }
  }

  return (
    <div className="effects-panel">
      <h3>Effects</h3>
      <div className="effects-tracks">
        {tracks.map(track => (
          <div key={track.id} className="effects-track">
            <div className="effects-track-header">
              <span>{track.name}</span>
              <div className="effects-add">
                <select 
                  onChange={(e) => {
                    if (e.target.value) {
                      handleAddEffect(track.id, e.target.value)
                      e.target.value = ''
                    }
                  }}
                >
                  <option value="">Add Effect...</option>
                  {availableEffects.map(effect => (
                    <option key={effect} value={effect}>{effect}</option>
                  ))}
                </select>
              </div>
            </div>
            <div className="effects-list">
              {track.effects?.map(effect => (
                <div key={effect.id} className="effect">
                  <div className="effect-header">
                    <span>{effect.type}</span>
                    <button 
                      className="remove-effect"
                      onClick={() => handleRemoveEffect(track.id, effect.id)}
                    >
                      <FaTimes />
                    </button>
                  </div>
                  <div className="effect-settings">
                    {Object.entries(effect.settings).map(([setting, value]) => (
                      <div key={setting} className="effect-setting">
                        <label>{setting}</label>
                        <input
                          type="range"
                          min="0"
                          max="1"
                          step="0.01"
                          value={value}
                          onChange={(e) => handleEffectSettingChange(
                            track.id,
                            effect.id,
                            setting,
                            parseFloat(e.target.value)
                          )}
                        />
                        <span>{value.toFixed(2)}</span>
                      </div>
                    ))}
                  </div>
                </div>
              ))}
            </div>
          </div>
        ))}
      </div>
    </div>
  )
}

export default EffectsPanel
