import React, { useState, useEffect } from 'react'
import TrackList from './TrackList'
import Transport from './Transport'
import Timeline from './Timeline'
import ToolBar from './ToolBar'
import Library from './Library'
import MixerPanel from './MixerPanel'
import EffectsPanel from './EffectsPanel'
import SampleBrowser from './SampleBrowser'
import ProjectSettingsPanel from './ProjectSettingsPanel'
import './Studio.css'

function Studio() {
  const [tracks, setTracks] = useState([
    { 
      id: 1, 
      name: 'Drums', 
      type: 'audio', 
      color: '#FF5252', 
      muted: false, 
      solo: false, 
      volume: 0, 
      pan: 0,
      effects: []
    },
    { 
      id: 2, 
      name: 'Bass', 
      type: 'audio', 
      color: '#448AFF', 
      muted: false, 
      solo: false, 
      volume: 0, 
      pan: 0,
      effects: []
    },
    { 
      id: 3, 
      name: 'Melody', 
      type: 'audio', 
      color: '#66BB6A', 
      muted: false, 
      solo: false, 
      volume: 0, 
      pan: 0,
      effects: []
    }
  ])

  const [isPlaying, setIsPlaying] = useState(false)
  const [isRecording, setIsRecording] = useState(false)
  const [currentTime, setCurrentTime] = useState(0)
  const [projectSettings, setProjectSettings] = useState({
    name: 'My Project',
    bpm: 120,
    timeSignature: '4/4'
  })
  const [showEffects, setShowEffects] = useState(false)
  const [showSampleBrowser, setShowSampleBrowser] = useState(false)
  const [showProjectSettings, setShowProjectSettings] = useState(false)
  const [collaborators] = useState([
    { id: 1, name: 'User 1', color: '#FF5252', cursor: 120 },
    { id: 2, name: 'User 2', color: '#448AFF', cursor: 240 }
  ])

  const addTrack = () => {
    const newTrack = {
      id: tracks.length + 1,
      name: `Track ${tracks.length + 1}`,
      type: 'audio',
      color: '#' + Math.floor(Math.random()*16777215).toString(16),
      muted: false,
      solo: false,
      volume: 0,
      pan: 0,
      effects: []
    }
    setTracks([...tracks, newTrack])
  }

  const handleUpdateTrack = (trackId, updates) => {
    setTracks(tracks.map(track => 
      track.id === trackId ? { ...track, ...updates } : track
    ))
  }

  const handleDropSample = (trackId, sample) => {
    console.log('Sample dropped:', sample, 'on track:', trackId)
    // Here we would add the sample to the track
  }

  const handleProjectSettingsChange = (newSettings) => {
    setProjectSettings(newSettings)
  }

  useEffect(() => {
    let animationFrameId
    const update = () => {
      const timeIncrement = (60 / projectSettings.bpm) * (1000 / 60) / 16.66
      if (isPlaying) {
        setCurrentTime((prevTime) => prevTime + timeIncrement)
      } else if (isRecording) {
        setCurrentTime((prevTime) => prevTime + timeIncrement)
      }
      animationFrameId = requestAnimationFrame(update)
    }

    if (isPlaying || isRecording) {
      animationFrameId = requestAnimationFrame(update)
    } else {
      cancelAnimationFrame(animationFrameId)
    }

    return () => cancelAnimationFrame(animationFrameId)
  }, [isPlaying, isRecording, projectSettings.bpm])

  return (
    <div className="studio">
      <ToolBar 
        onAddTrack={addTrack} 
        onToggleEffects={() => setShowEffects(!showEffects)} 
        onToggleSampleBrowser={() => setShowSampleBrowser(!showSampleBrowser)}
        onToggleProjectSettings={() => setShowProjectSettings(!showProjectSettings)}
      />
      
      <div className="studio-main">
        <div className="studio-sidebar">
          <Library />
          {showSampleBrowser && (
            <SampleBrowser onSampleSelect={(sample) => console.log('Sample selected:', sample)} />
          )}
        </div>
        
        <div className="studio-content">
          <Timeline 
            currentTime={currentTime} 
            collaborators={collaborators}
          />
          
          <TrackList 
            tracks={tracks} 
            setTracks={setTracks}
            currentTime={currentTime}
            onDropSample={handleDropSample}
          />

          <MixerPanel tracks={tracks} onUpdateTrack={handleUpdateTrack} />
          
          <Transport 
            isPlaying={isPlaying}
            setIsPlaying={setIsPlaying}
            isRecording={isRecording}
            setIsRecording={setIsRecording}
            currentTime={currentTime}
            setCurrentTime={setCurrentTime}
            bpm={projectSettings.bpm}
            setBpm={(newBpm) => handleProjectSettingsChange({ ...projectSettings, bpm: newBpm })}
          />
        </div>

        {showEffects && (
          <div className="studio-effects">
            <EffectsPanel tracks={tracks} onUpdateTrack={handleUpdateTrack} />
          </div>
        )}

        {showProjectSettings && (
          <div className="studio-project-settings">
            <ProjectSettingsPanel settings={projectSettings} onSettingsChange={handleProjectSettingsChange} />
          </div>
        )}
      </div>
    </div>
  )
}

export default Studio
